/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelContextAware;
import org.apache.camel.Consumer;
import org.apache.camel.Route;
import org.apache.camel.Service;
import org.apache.camel.ShutdownRoute;
import org.apache.camel.ShutdownRunningTask;
import org.apache.camel.SuspendableService;
import org.apache.camel.spi.RouteStartupOrder;
import org.apache.camel.spi.ShutdownAware;
import org.apache.camel.spi.ShutdownStrategy;
import org.apache.camel.support.ServiceSupport;
import org.apache.camel.util.EventHelper;
import org.apache.camel.util.ObjectHelper;
import org.apache.camel.util.ServiceHelper;
import org.apache.camel.util.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultShutdownStrategy
extends ServiceSupport
implements ShutdownStrategy,
CamelContextAware {
    private static final transient Logger LOG = LoggerFactory.getLogger(DefaultShutdownStrategy.class);
    private CamelContext camelContext;
    private ExecutorService executor;
    private long timeout = 300L;
    private TimeUnit timeUnit = TimeUnit.SECONDS;
    private boolean shutdownNowOnTimeout = true;
    private boolean shutdownRoutesInReverseOrder = true;
    private volatile boolean forceShutdown;

    public DefaultShutdownStrategy() {
    }

    public DefaultShutdownStrategy(CamelContext camelContext) {
        this.camelContext = camelContext;
    }

    @Override
    public void shutdown(CamelContext context, List<RouteStartupOrder> routes) throws Exception {
        this.shutdown(context, routes, this.getTimeout(), this.getTimeUnit());
    }

    @Override
    public void shutdownForced(CamelContext context, List<RouteStartupOrder> routes) throws Exception {
        this.doShutdown(context, routes, this.getTimeout(), this.getTimeUnit(), false, false, true);
    }

    @Override
    public void suspend(CamelContext context, List<RouteStartupOrder> routes) throws Exception {
        this.doShutdown(context, routes, this.getTimeout(), this.getTimeUnit(), true, false, false);
    }

    @Override
    public void shutdown(CamelContext context, List<RouteStartupOrder> routes, long timeout, TimeUnit timeUnit) throws Exception {
        this.doShutdown(context, routes, timeout, timeUnit, false, false, false);
    }

    @Override
    public boolean shutdown(CamelContext context, RouteStartupOrder route, long timeout, TimeUnit timeUnit, boolean abortAfterTimeout) throws Exception {
        ArrayList<RouteStartupOrder> routes = new ArrayList<RouteStartupOrder>(1);
        routes.add(route);
        return this.doShutdown(context, routes, timeout, timeUnit, false, abortAfterTimeout, false);
    }

    @Override
    public void suspend(CamelContext context, List<RouteStartupOrder> routes, long timeout, TimeUnit timeUnit) throws Exception {
        this.doShutdown(context, routes, timeout, timeUnit, true, false, false);
    }

    protected boolean doShutdown(CamelContext context, List<RouteStartupOrder> routes, long timeout, TimeUnit timeUnit, boolean suspendOnly, boolean abortAfterTimeout, boolean forceShutdown) throws Exception {
        StopWatch watch = new StopWatch();
        ArrayList<RouteStartupOrder> routesOrdered = new ArrayList<RouteStartupOrder>(routes);
        Collections.sort(routesOrdered, new Comparator<RouteStartupOrder>(){

            @Override
            public int compare(RouteStartupOrder o1, RouteStartupOrder o2) {
                return o1.getStartupOrder() - o2.getStartupOrder();
            }
        });
        if (this.shutdownRoutesInReverseOrder) {
            Collections.reverse(routesOrdered);
        }
        if (timeout > 0L) {
            LOG.info("Starting to graceful shutdown " + routesOrdered.size() + " routes (timeout " + timeout + " " + timeUnit.toString().toLowerCase() + ")");
        } else {
            LOG.info("Starting to graceful shutdown " + routesOrdered.size() + " routes (no timeout)");
        }
        Future<?> future = this.getExecutorService().submit(new ShutdownTask(context, routesOrdered, timeout, timeUnit, suspendOnly, abortAfterTimeout));
        try {
            if (timeout > 0L) {
                future.get(timeout, timeUnit);
            } else {
                future.get();
            }
        }
        catch (TimeoutException e) {
            future.cancel(true);
            this.forceShutdown = forceShutdown;
            if (!forceShutdown && abortAfterTimeout) {
                LOG.warn("Timeout occurred. Aborting the shutdown now.");
                return false;
            }
            if (forceShutdown || this.shutdownNowOnTimeout) {
                LOG.warn("Timeout occurred. Now forcing the routes to be shutdown now.");
                this.shutdownRoutesNow(routesOrdered);
            } else {
                LOG.warn("Timeout occurred. Will ignore shutting down the remainder routes.");
            }
        }
        catch (ExecutionException e) {
            throw ObjectHelper.wrapRuntimeCamelException(e.getCause());
        }
        long seconds = TimeUnit.SECONDS.convert(watch.stop(), TimeUnit.MILLISECONDS);
        LOG.info("Graceful shutdown of " + routesOrdered.size() + " routes completed in " + seconds + " seconds");
        return true;
    }

    @Override
    public boolean forceShutdown(Service service) {
        return this.forceShutdown;
    }

    @Override
    public void setTimeout(long timeout) {
        this.timeout = timeout;
    }

    @Override
    public long getTimeout() {
        return this.timeout;
    }

    @Override
    public void setTimeUnit(TimeUnit timeUnit) {
        this.timeUnit = timeUnit;
    }

    @Override
    public TimeUnit getTimeUnit() {
        return this.timeUnit;
    }

    @Override
    public void setShutdownNowOnTimeout(boolean shutdownNowOnTimeout) {
        this.shutdownNowOnTimeout = shutdownNowOnTimeout;
    }

    @Override
    public boolean isShutdownNowOnTimeout() {
        return this.shutdownNowOnTimeout;
    }

    @Override
    public boolean isShutdownRoutesInReverseOrder() {
        return this.shutdownRoutesInReverseOrder;
    }

    @Override
    public void setShutdownRoutesInReverseOrder(boolean shutdownRoutesInReverseOrder) {
        this.shutdownRoutesInReverseOrder = shutdownRoutesInReverseOrder;
    }

    @Override
    public CamelContext getCamelContext() {
        return this.camelContext;
    }

    @Override
    public void setCamelContext(CamelContext camelContext) {
        this.camelContext = camelContext;
    }

    protected void shutdownRoutesNow(List<RouteStartupOrder> routes) {
        for (RouteStartupOrder order : routes) {
            ShutdownRunningTask current = order.getRoute().getRouteContext().getShutdownRunningTask();
            if (current != ShutdownRunningTask.CompleteCurrentTaskOnly) {
                LOG.debug("Changing shutdownRunningTask from {} to " + (Object)((Object)ShutdownRunningTask.CompleteCurrentTaskOnly) + " on route {} to shutdown faster", (Object)current, (Object)order.getRoute().getId());
                order.getRoute().getRouteContext().setShutdownRunningTask(ShutdownRunningTask.CompleteCurrentTaskOnly);
            }
            for (Consumer consumer : order.getInputs()) {
                DefaultShutdownStrategy.shutdownNow(consumer);
            }
        }
    }

    protected void shutdownNow(List<Consumer> consumers) {
        for (Consumer consumer : consumers) {
            DefaultShutdownStrategy.shutdownNow(consumer);
        }
    }

    protected static void shutdownNow(Consumer consumer) {
        LOG.trace("Shutting down: {}", consumer);
        try {
            ServiceHelper.stopService(consumer);
        }
        catch (Throwable e) {
            LOG.warn("Error occurred while shutting down route: " + consumer + ". This exception will be ignored.", e);
            EventHelper.notifyServiceStopFailure(consumer.getEndpoint().getCamelContext(), consumer, e);
        }
        LOG.trace("Shutdown complete for: {}", consumer);
    }

    protected static void suspendNow(Consumer consumer) {
        LOG.trace("Suspending: {}", consumer);
        try {
            ServiceHelper.suspendService(consumer);
        }
        catch (Throwable e) {
            LOG.warn("Error occurred while suspending route: " + consumer + ". This exception will be ignored.", e);
            EventHelper.notifyServiceStopFailure(consumer.getEndpoint().getCamelContext(), consumer, e);
        }
        LOG.trace("Suspend complete for: {}", consumer);
    }

    private ExecutorService getExecutorService() {
        if (this.executor == null) {
            this.executor = this.camelContext.getExecutorServiceManager().newSingleThreadExecutor(this, "ShutdownTask");
        }
        return this.executor;
    }

    @Override
    protected void doStart() throws Exception {
        ObjectHelper.notNull(this.camelContext, "CamelContext");
        this.forceShutdown = false;
    }

    @Override
    protected void doStop() throws Exception {
    }

    @Override
    protected void doShutdown() throws Exception {
        if (this.executor != null) {
            this.camelContext.getExecutorServiceManager().shutdownNow(this.executor);
            this.executor = null;
        }
    }

    static class ShutdownTask
    implements Runnable {
        private final CamelContext context;
        private final List<RouteStartupOrder> routes;
        private final boolean suspendOnly;
        private final boolean abortAfterTimeout;
        private final long timeout;
        private final TimeUnit timeUnit;

        public ShutdownTask(CamelContext context, List<RouteStartupOrder> routes, long timeout, TimeUnit timeUnit, boolean suspendOnly, boolean abortAfterTimeout) {
            this.context = context;
            this.routes = routes;
            this.suspendOnly = suspendOnly;
            this.abortAfterTimeout = abortAfterTimeout;
            this.timeout = timeout;
            this.timeUnit = timeUnit;
        }

        @Override
        public void run() {
            LOG.debug("There are {} routes to {}", this.routes.size(), (Object)(this.suspendOnly ? "suspend" : "shutdown"));
            ArrayList<ShutdownDeferredConsumer> deferredConsumers = new ArrayList<ShutdownDeferredConsumer>();
            for (RouteStartupOrder order : this.routes) {
                ShutdownRoute shutdownRoute = order.getRoute().getRouteContext().getShutdownRoute();
                ShutdownRunningTask shutdownRunningTask = order.getRoute().getRouteContext().getShutdownRunningTask();
                if (LOG.isTraceEnabled()) {
                    LOG.trace("{}{} with options [{},{}]", new Object[]{this.suspendOnly ? "Suspending route: " : "Shutting down route: ", order.getRoute().getId(), shutdownRoute, shutdownRunningTask});
                }
                for (Consumer consumer : order.getInputs()) {
                    boolean shutdown;
                    boolean suspend = false;
                    boolean bl = shutdown = shutdownRoute != ShutdownRoute.Defer;
                    if (shutdown) {
                        if (consumer instanceof ShutdownAware) {
                            boolean bl2 = shutdown = !((ShutdownAware)((Object)consumer)).deferShutdown(shutdownRunningTask);
                        }
                        if (shutdown && consumer instanceof SuspendableService) {
                            suspend = true;
                        }
                    }
                    if (suspend) {
                        DefaultShutdownStrategy.suspendNow(consumer);
                        deferredConsumers.add(new ShutdownDeferredConsumer(order.getRoute(), consumer));
                        LOG.debug("Route: {} suspended and shutdown deferred, was consuming from: {}", (Object)order.getRoute().getId(), (Object)order.getRoute().getEndpoint());
                        continue;
                    }
                    if (shutdown) {
                        DefaultShutdownStrategy.shutdownNow(consumer);
                        LOG.info("Route: {} shutdown complete, was consuming from: {}", (Object)order.getRoute().getId(), (Object)order.getRoute().getEndpoint());
                        continue;
                    }
                    deferredConsumers.add(new ShutdownDeferredConsumer(order.getRoute(), consumer));
                    LOG.debug("Route: " + order.getRoute().getId() + (this.suspendOnly ? " shutdown deferred." : " suspension deferred."));
                }
            }
            boolean done = false;
            long loopDelaySeconds = 1L;
            long loopCount = 0L;
            while (!done) {
                int size = 0;
                for (RouteStartupOrder order : this.routes) {
                    int inflight = this.context.getInflightRepository().size(order.getRoute().getId());
                    for (Consumer consumer : order.getInputs()) {
                        if (!(consumer instanceof ShutdownAware)) continue;
                        inflight += ((ShutdownAware)((Object)consumer)).getPendingExchangesSize();
                    }
                    if (inflight <= 0) continue;
                    size += inflight;
                    LOG.trace("{} inflight and pending exchanges for route: {}", inflight, (Object)order.getRoute().getId());
                }
                if (size > 0) {
                    try {
                        LOG.info("Waiting as there are still " + size + " inflight and pending exchanges to complete, timeout in " + (TimeUnit.SECONDS.convert(this.timeout, this.timeUnit) - loopCount++ * loopDelaySeconds) + " seconds.");
                        Thread.sleep(loopDelaySeconds * 1000L);
                        continue;
                    }
                    catch (InterruptedException e) {
                        if (this.abortAfterTimeout) {
                            LOG.warn("Interrupted while waiting during graceful shutdown, will abort.");
                            return;
                        }
                        LOG.warn("Interrupted while waiting during graceful shutdown, will force shutdown now.");
                        break;
                    }
                }
                done = true;
            }
            for (ShutdownDeferredConsumer deferred : deferredConsumers) {
                Consumer consumer = deferred.getConsumer();
                if (!(consumer instanceof ShutdownAware)) continue;
                LOG.trace("Route: {} preparing to shutdown.", (Object)deferred.getRoute().getId());
                ((ShutdownAware)((Object)consumer)).prepareShutdown();
                LOG.debug("Route: {} preparing to shutdown complete.", (Object)deferred.getRoute().getId());
            }
            for (ShutdownDeferredConsumer deferred : deferredConsumers) {
                Consumer consumer = deferred.getConsumer();
                if (this.suspendOnly) {
                    DefaultShutdownStrategy.suspendNow(consumer);
                    LOG.info("Route: {} suspend complete, was consuming from: {}", (Object)deferred.getRoute().getId(), (Object)deferred.getConsumer().getEndpoint());
                    continue;
                }
                DefaultShutdownStrategy.shutdownNow(consumer);
                LOG.info("Route: {} shutdown complete, was consuming from: {}", (Object)deferred.getRoute().getId(), (Object)deferred.getConsumer().getEndpoint());
            }
        }
    }

    static class ShutdownDeferredConsumer {
        private final Route route;
        private final Consumer consumer;

        ShutdownDeferredConsumer(Route route, Consumer consumer) {
            this.route = route;
            this.consumer = consumer;
        }

        Route getRoute() {
            return this.route;
        }

        Consumer getConsumer() {
            return this.consumer;
        }
    }
}

