package org.modelbus.library.example.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import library.Library;
import library.LibraryFactory;
import library.Writer;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.modelbus.core.lib.ModelBusCoreLib;
import org.modelbus.dosgi.repository.descriptor.ConstraintViolationException;
import org.modelbus.dosgi.repository.descriptor.InvalidValueException;
import org.modelbus.dosgi.repository.descriptor.LockedException;
import org.modelbus.dosgi.repository.descriptor.NonExistingResourceException;
import org.modelbus.dosgi.repository.descriptor.Property;
import org.modelbus.dosgi.repository.descriptor.RepositoryAuthentificationException;
import org.modelbus.dosgi.repository.descriptor.Session;
import org.modelbus.dosgi.repository.descriptor.UnresolvedReferencesException;

public class LibraryServiceImpl implements LibraryService {
	private OsloOCLEvaluator evaluator;
	
	public LibraryServiceImpl(){
		this.evaluator = new OsloOCLEvaluator();
	}
	
	private final static String[] OCL_EXPRESSIONS = new String[]{
		"context Library def: getWriters() : Set(Writer) = self.writers",
		"context Library def: getLooseWriters() : Set(Writer) = self.writers->select(w:Writer | w.books->isEmpty())"
	};
	
	private final static String OCL_WRITER_VALID = "context Writer inv: self.name.size() > 0";
	
	@Override
	public void addWriter(final Library library, final String name) {
		final Writer writer = LibraryFactory.eINSTANCE.createWriter();
		writer.setName(name);
		
		library.getWriters().add(writer);

		final Resource res = library.eResource();
		
		try {
			ModelBusCoreLib.getRepositoryHelper().checkInModel(this.getSession(), res, res.getURI(),Collections.EMPTY_MAP, "Library service commit");
		} catch (RepositoryAuthentificationException e) {
			throw new RuntimeException(e);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (NonExistingResourceException e) {
			throw new RuntimeException(e);
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		} catch (ConstraintViolationException e) {
			throw new RuntimeException(e);
		} catch (LockedException e) {
			throw new RuntimeException(e);
		} catch (UnresolvedReferencesException e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public List<Writer> getLooseWriters(Library library) {
		evaluator.addModel(library.eClass().getEPackage());
		
		this.initializeOCLProcessor();
		
		final String ocl = "context Library inv: self.getLooseWriters()";
		
		final List<Object> results = this.evaluator.evaluateExpression(library, ocl);
		
		return toWriterList(results);
	}

	@Override
	public List<Writer> getWriters(final Library library) {
		evaluator.addModel(library.eClass().getEPackage());
		
		this.initializeOCLProcessor();
		
		final String ocl = "context Library inv: self.getWriters()";
		
		final List<Object> results = this.evaluator.evaluateExpression(library, ocl);
		
		return toWriterList(results);
	}
	
	@Override
	public boolean isLibraryModelValid(final Library library) {
		evaluator.addModel(library.eClass().getEPackage());
		
		boolean valid = true;
		
		//check if library contains writers
		final List<Object> results1 = this.evaluator.evaluateExpression(library, "context Library inv: self.writers->notEmpty()");
		
		if((Boolean)results1.get(0) == false){
			System.out.println("Model is invalid: no writers");
			valid = false;
		}
		
		//check if library contains not more than 3 writers
		final List<Object> results2 = this.evaluator.evaluateExpression(library, "context Library inv: self.writers->size() < 4");
		
		if((Boolean)results2.get(0) == false){
			System.out.println("Model is invalid: more than 3 writers");
			valid = false;
		}
		
		//check if all writer names are not empty
		final List<Object> results3 = this.evaluator.evaluateExpression(library, "context Library inv: self.writers->select(w:Writer | w.name.size() = 0)->isEmpty()");
		
		if((Boolean)results3.get(0) == false){
			System.out.println("Model is invalid: empty writer name detected");
			valid = false;
		}
		
		return valid;
	}
	
	private void initializeOCLProcessor(){
		for(final String oclExpression : OCL_EXPRESSIONS){
			this.evaluator.createOCLOperation(oclExpression);
		}
	}
	
	private Session getSession(){
		Session session = new Session();
		session.setId(EcoreUtil.generateUUID());
		Property propertyUserName = new Property();
		propertyUserName.setKey("username");
		propertyUserName.setValue("Admin");
		Property propertyPassword = new Property();
		propertyPassword.setKey("password");
		propertyPassword.setValue("ModelBus");

		session.getProperties().add(propertyUserName);
		session.getProperties().add(propertyPassword);
		
		return session;
	}
	
	@SuppressWarnings("unchecked")
	private static List<Writer> toWriterList(List<Object> list){
		final List<Writer> writers = new ArrayList<Writer>();
		
		final List<Object> subList = (List<Object>)list.get(0);
		
		for(final Object obj : subList){
			writers.add((Writer)obj);
		}
		
		return writers;
	}
}
